/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Drawing;
using System.Collections;
using System.Collections.Specialized;
using System.ComponentModel.Design;
using System.ComponentModel;
using System.Threading;
using System.Globalization;

using Borland.Eco.Internal.DefaultImpl;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Persistence.Multi;
using Borland.Eco.UmlRt;
using Borland.Eco.Services;
using Borland.Eco.Globalization;
using Borland.Eco.Exceptions;

namespace Borland.Eco.Persistence
{

	internal enum CreateMappingMode
	{
		CreateDb,
		Run
	};
	public enum TransactionMode
	{
		Disabled,
		Local,
		Distributed
	};
	[ToolboxBitmap(typeof(PersistenceMapperMultiDb), "Borland.Eco.Persistence.PersistenceMapperMultiDb.bmp")]
	[ToolboxItem(true)]
	public class PersistenceMapperMultiDb: AbstractPersistenceMapperDb
	{
		public PersistenceMapperMultiDb()
		{
			TransactionMode = TransactionMode.Distributed;
		}

		public override void CloseConnections(bool force)
		{
			foreach (PMapperDef def in PersistenceMappers)
				if (def.Mapper != null)
					def.Mapper.ConnectionPool.CloseConnections(force);
		}

		public IDatabaseCollection RetrieveDatabaseConnections()
		{
			effectiveTransactionManager = DistributedTransactionManager;
			if (effectiveTransactionManager == null)
			{
				if (TransactionMode == TransactionMode.Distributed)
					effectiveTransactionManager = new DistributedTransactionManager();
				else if (TransactionMode == TransactionMode.Local)
					effectiveTransactionManager = new LocalTransactionManager();
			}

			Borland.Eco.Persistence.Connection.DatabaseCollection res = new Borland.Eco.Persistence.Connection.DatabaseCollection(effectiveTransactionManager);
			foreach (PMapperDef def in PersistenceMappers)
			{
				IDatabase db = def.Mapper.ConnectionPool.RetrieveDatabaseConnection(true);
				res.Add(db, def.Name);
			}
			return res;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="databases"/> is null</exception>
		public void ReturnDatabaseConnections(IDatabaseCollection databases)
		{
			if (databases == null) throw new ArgumentNullException("databases"); // do not localize
			foreach (string name in databases.GetNames())
			{
				PMapperDef def = PersistenceMappers.FindByName(name);
				def.Mapper.ConnectionPool.ReturnDatabaseConnection(databases.GetDatabase(name));
			}
		}

		public void CreateDatabaseSchema(ITypeSystemService typeSystemService, string dbName)
		{
			CreateDatabaseSchema(typeSystemService, new DefaultCleanPsConfig(false), dbName);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="configureCleanPS"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="typeSystemService"/> is null.</exception>
		///<exception cref="ArgumentException">Thrown if the no definition is found for <paramref name="dbName"/>.</exception>
		public void CreateDatabaseSchema(ITypeSystemService typeSystemService, IConfigureCleanPS configureCleanPS, string dbName)
		{
			if (configureCleanPS == null) throw new ArgumentNullException("configureCleanPS"); // do not localize
			if (typeSystemService == null) throw new ArgumentNullException("typeSystemService"); // do not localize

			PMapperDef def = PersistenceMappers.FindByName(dbName);
			if (def == null)
				throw new ArgumentException(MultiStringRes.sNoDatabaseWithName(dbName));
			IDatabaseCollection databases = RetrieveDatabaseConnections();
			try
			{
				ORMappingDefinition mapping = GetCompoundMapping(typeSystemService, databases, CreateMappingMode.CreateDb);
				SimpleORMappingProvider newMappingProvider = new SimpleORMappingProvider(mapping, def.Mapper.EffectiveNewMappingProvider);
				PersistenceHandleDb.CreateDataBaseSchema(typeSystemService,
					databases,
					def.Name,
					newMappingProvider,
					def.Mapper.EffectiveOldMappingProvider, configureCleanPS);
			}
			finally
			{
				ReturnDatabaseConnections(databases);
			}
		}

		private ORMappingDefinition GetCompoundMapping(ITypeSystemService typeSystemService, IDatabaseCollection databases, CreateMappingMode mode)
		{
			ORMappingDefinition mergedMapping = new ORMappingDefinition();
			if (RunTimeMappingProvider != null)
			{
				MergeMapping(typeSystemService, mergedMapping, RunTimeMappingProvider, null, null, "", mode);
			}
			foreach (PMapperDef def in PersistenceMappers)
			{
				if (def.Mapper != null && def.ImportConfig)
				{
					IORMappingProvider mappingProvider = def.Mapper.RunTimeMappingProvider;
					if (mappingProvider == null)
					{
						DefaultORMappingBuilder implicitMappingProvider = new DefaultORMappingBuilder();
						implicitMappingProvider.Database = def.Name;
						mappingProvider = implicitMappingProvider;
						def.Mapper.RunTimeMappingProvider = mappingProvider;
					}

					IDatabase db = databases.GetDatabase(def.Name);
					MergeMapping(typeSystemService, mergedMapping,
						mappingProvider,
						(mode == CreateMappingMode.Run) ? db : null,
						db.Config,
						def.Name,
						mode);
				}
			}

			ColumnTypeFixer.FixTypes(mergedMapping, null, typeSystemService.TypeSystem);

			mergedMapping.Validate(null, typeSystemService.TypeSystem);
			return mergedMapping;
		}

		private static void MergeMapping(ITypeSystemService typeSystemService,
			ORMappingDefinition targetMapping,
			IORMappingProvider sourceMappingProvider,
			IDatabase db,
			SqlDatabaseConfig config,
			string dbName,
			CreateMappingMode mode)
		{
			sourceMappingProvider.Initialize(typeSystemService.TypeSystem, db, config, mode == CreateMappingMode.CreateDb);

			if (config != null)
				ColumnTypeFixer.FixTypes(sourceMappingProvider.Mapping, config, typeSystemService.TypeSystem);
			targetMapping.Merge(sourceMappingProvider.Mapping, dbName);
		}

		///<exception cref="InvalidOperationException">Thrown if the mapper is used with another type system.</exception>
		public override IPersistenceMapper GetPersistenceMapper(ITypeSystemService typeSystemService)
		{
			lock(this)
			{
				if ((PmapperWithDb == null) || (this.TypeSystemService != typeSystemService))
				{
					if (ActiveCount > 0)
						throw new InvalidOperationException(PersistenceStringRes.sPersistenceMapperInUseWithOtherTypeSystem);

					SetPersistenceMapperWithIDatabase(null);
					SetTypeSystemService(null);
					IDatabaseCollection databases = RetrieveDatabaseConnections();

					ORMappingDefinition mergedMapping = GetCompoundMapping(typeSystemService, databases, CreateMappingMode.Run);

					try
					{
						SetTypeSystemService(typeSystemService);
						SetPersistenceMapperWithIDatabase(PersistenceHandleDb.GetPersistenceMapper(
							TypeSystemService,
							databases,
							mergedMapping,
							VersionGranularity,
							OnGetCurrentTime));
					}
					finally
					{
						ReturnDatabaseConnections(databases);
					}
				}
				ActiveCount++;
				return new PersistenceMapperMultiDbAdapter(PmapperWithDb, this);
			}
		}

		public override void ReturnPersistenceMapper(IPersistenceMapper persistenceMapper)
		{
			lock(this)
			{
				ActiveCount--;
				if (ActiveCount == 0)
					CloseConnections(false);
			}
		}

		private PersistenceMapperDbCollection m_PersistenceMappers = new PersistenceMapperDbCollection();
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		public PersistenceMapperDbCollection PersistenceMappers { get { return m_PersistenceMappers; } }

		private ITransactionManager m_DistributedTransactionManager;
		///<summary>
		/// Setting this property will override the TransactionMode property.
		/// To use another transactionmanager than MS DTC, you need an instance of a class that implements
		/// ITransactionManager.
		///</summary>
		[Browsable (false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public ITransactionManager DistributedTransactionManager
		{
			get { return m_DistributedTransactionManager; }
			set { m_DistributedTransactionManager = value; }
		}
		private ITransactionManager effectiveTransactionManager;
		[DefaultValue (TransactionMode.Distributed)]
		private TransactionMode m_TransactionMode;
		///<summary>
		/// This property controls how transactions are handled when updates are perfomed
		/// * Disabled - No transactions are performed by Eco
		/// * Local - Only local transactions are used with each participating database. If one database fails to commit, others may already have committed, thus creating an inconsistency!
		/// * Distributed - Use Microsoft DTC to handle distributed transactions. This option also supports declarative transactions.
		///</summary>
		public TransactionMode TransactionMode
		{
			get { return m_TransactionMode; }
			set { m_TransactionMode = value; }
		}

		private sealed class PersistenceMapperMultiDbAdapter: MarshalByRefObject, IPersistenceMapper
		{
			private IPersistenceMapperWithIDatabase m_Adaptee;
			private readonly PersistenceMapperMultiDb m_PersistenceMapperMultiDb;

			private IDatabase RetrieveVersionedDatabaseConnection()
			{
				PMapperDef def = m_PersistenceMapperMultiDb.PersistenceMappers.FindVersionedDatabase();
				return def.Mapper.ConnectionPool.RetrieveDatabaseConnection(false);
			}

			private IDatabaseCollection RetrieveDatabaseConnections()
			{
				return m_PersistenceMapperMultiDb.RetrieveDatabaseConnections();
			}

			private void ReturnDatabaseConnections(IDatabaseCollection databases)
			{
				m_PersistenceMapperMultiDb.ReturnDatabaseConnections(databases);
			}

			private void ReturnVersionedDatabaseConnection(IDatabase database)
			{
				PMapperDef def = m_PersistenceMapperMultiDb.PersistenceMappers.FindVersionedDatabase();
				def.Mapper.ConnectionPool.ReturnDatabaseConnection(database);
			}

			///<exception cref="InvalidOperationException">Thrown if the adapter is deactivated.</exception>
			private IPersistenceMapperWithIDatabase Adaptee
			{
				get
				{
					if (m_Adaptee == null)
						throw new InvalidOperationException("Attempt to use deactivated PersistenceMapperAdapter"); // Do not localize
					return m_Adaptee;
				}
			}

			public PersistenceMapperMultiDbAdapter(
				IPersistenceMapperWithIDatabase persistenceMapperWithIDatabase,
				PersistenceMapperMultiDb persistenceMapperMultiDb)
			{
				m_PersistenceMapperMultiDb = persistenceMapperMultiDb;
				m_Adaptee = persistenceMapperWithIDatabase;
			}

			public void DeActivate()
			{
				m_Adaptee = null;
			}

			#region IPersistenceMapper implementation

			void IPersistenceMapper.Fetch(ObjectIdList idList, out Datablock datablock, int[] memberIdList)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.Fetch(databases, idList, out datablock, memberIdList);
					datablock.Strip();
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}
			void IPersistenceMapper.FetchIDListWithCondition(AbstractCondition condition, out ObjectIdList result, int maxResults, int offset)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.FetchIDListWithCondition(databases, condition, out result, maxResults, offset);
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}
			void IPersistenceMapper.VersionForTime(DateTime clockTime, out int version)
			{
				IDatabase database = RetrieveVersionedDatabaseConnection();
				try
				{
					Adaptee.VersionForTime(database, clockTime, out version);
				}
				finally
				{
					ReturnVersionedDatabaseConnection(database);
				}
			}
			void IPersistenceMapper.TimeForVersion(int version, out DateTime clockTime)
			{
				IDatabase database =  RetrieveVersionedDatabaseConnection();
				try
				{
					Adaptee.TimeForVersion(database, version, out clockTime);
				}
				finally
				{
					ReturnVersionedDatabaseConnection(database);
				}
			}

			bool IPersistenceMapper.SupportsSync
			{
				get { return m_PersistenceMapperMultiDb.SyncHandler != null; }
			}

			void IPersistenceMapper.Update(Datablock datablock, UpdatePrecondition precondition, out IdTranslationList translationList, out int version, out SyncVersion syncVersion, out UpdateResult result)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.Update(databases, datablock, precondition, out translationList, out version, out result, out syncVersion,
					 m_PersistenceMapperMultiDb.SyncHandler);
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}

			void IPersistenceMapper.GetChangesSince(SyncVersion syncVersion, SyncVersion[] excludeList, out DBChangeCollection changes, out SyncVersion lastSyncVersion)
			{
				if (m_PersistenceMapperMultiDb.SyncHandler == null)
					throw new InvalidOperationException(PersistenceStringRes.sSyncNotActive);
				changes = m_PersistenceMapperMultiDb.SyncHandler.GetChangesSince(syncVersion, excludeList, out lastSyncVersion, m_PersistenceMapperMultiDb.TypeSystemService.TypeSystem);
			}

			SyncVersion IPersistenceMapper.CurrentSyncVersion
			{
				get
				{
				if (m_PersistenceMapperMultiDb.SyncHandler == null)
					throw new InvalidOperationException(PersistenceStringRes.sSyncNotActive);
				return m_PersistenceMapperMultiDb.SyncHandler.CurrentSyncVersion();
				}
			}
			#endregion
		}
	}

	public sealed class SimpleORMappingProvider: IORMappingProvider
	{

		ORMappingDefinition m_Mapping;
		IORMappingProvider m_NextProvider;
		public SimpleORMappingProvider(ORMappingDefinition mapping, IORMappingProvider nextProvider)
		{
			m_Mapping = mapping;
			m_NextProvider = nextProvider;
		}
		public ORMappingDefinition Mapping { get { return m_Mapping; } }
		public void Initialize(IEcoTypeSystem typeSystem, IDatabase db, SqlDatabaseConfig config, bool addMissingTypes)
		{
			// intentionally left blank
		}

		public StringCollection SystemTableNames(SqlDatabaseConfig config)
		{
			return m_NextProvider.SystemTableNames(config);
		}

		public void PostCreateDb(IDatabase db, SqlDatabaseConfig sqlDatabaseConfig, ORMappingDefinition newMapping)
		{
			m_NextProvider.PostCreateDb(db, sqlDatabaseConfig, newMapping);
		}
		public void PostEvolveDb(IDatabase db, SqlDatabaseConfig sqlDatabaseConfig, ORMappingDefinition newMapping)
		{
			m_NextProvider.PostEvolveDb(db, sqlDatabaseConfig, newMapping);
		}
		public void SaveMappingInfo(IDatabase db, SqlDatabaseConfig sqlDatabaseConfig, ORMappingDefinition newMapping)
		{
			m_NextProvider.SaveMappingInfo(db, sqlDatabaseConfig, newMapping);
		}
	}
}